import express from 'express';
import Division from '../models/Division.js';
import { authenticateToken, authorizeRole } from '../middleware/auth.js';

const router = express.Router();

// Get all divisions
router.get('/', authenticateToken, async (req, res) => {
  try {
    const divisions = await Division.find({});
    res.json(divisions);
  } catch (error) {
    res.status(500).json({ error: 'Failed to fetch divisions' });
  }
});
// Add this new route
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const division = await Division.findById(req.params.id);
    if (!division) {
      return res.status(404).json({ error: 'Division not found' });
    }
    res.json(division);
  } catch (error) {
    res.status(500).json({ error: 'Failed to fetch division' });
  }
});

// Create a new division (admin-only)
router.post('/', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const { name } = req.body;
    const newDivision = new Division({ name });
    await newDivision.save();
    res.status(201).json(newDivision);
  } catch (error) {
    if (error.code === 11000) {
      return res.status(400).json({ error: 'Division with this name already exists' });
    }
    res.status(500).json({ error: 'Failed to create division' });
  }
});

// Update a division by ID (admin-only)
router.put('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const { name } = req.body;
    const updatedDivision = await Division.findByIdAndUpdate(
      req.params.id,
      { name },
      { new: true, runValidators: true }
    );
    if (!updatedDivision) {
      return res.status(404).json({ error: 'Division not found' });
    }
    res.json(updatedDivision);
  } catch (error) {
    if (error.code === 11000) {
      return res.status(400).json({ error: 'Division with this name already exists' });
    }
    res.status(500).json({ error: 'Failed to update division' });
  }
});

// Delete a division by ID (admin-only)
router.delete('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const division = await Division.findByIdAndDelete(req.params.id);
    if (!division) {
      return res.status(404).json({ error: 'Division not found' });
    }
    res.json({ message: 'Division deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: 'Failed to delete division' });
  }
});

export default router;