import express from 'express';
import Subdivision from '../models/Subdivision.js';
import { authenticateToken, authorizeRole } from '../middleware/auth.js';

const router = express.Router();

// Get all subdivisions
// Optionally, filter by division ID
router.get('/', authenticateToken, async (req, res) => {
  try {
    const { division } = req.query;
    let query = {};
    if (division) {
      query.division = division;
    }
    const subdivisions = await Subdivision.find(query).populate('division', 'name');
    res.json(subdivisions);
  } catch (error) {
    res.status(500).json({ error: 'Failed to fetch subdivisions' });
  }
});
// Add this new route
router.get('/:id', authenticateToken, async (req, res) => {
  try {
    const subdivision = await Subdivision.findById(req.params.id);
    if (!subdivision) {
      return res.status(404).json({ error: 'Subdivision not found' });
    }
    res.json(subdivision);
  } catch (error) {
    res.status(500).json({ error: 'Failed to fetch subdivision' });
  }
});

// Create a new subdivision (admin-only)
router.post('/', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const { name, division } = req.body;
    const newSubdivision = new Subdivision({ name, division });
    await newSubdivision.save();
    res.status(201).json(newSubdivision);
  } catch (error) {
    if (error.code === 11000) {
      return res.status(400).json({ error: 'Subdivision with this name already exists' });
    }
    res.status(500).json({ error: 'Failed to create subdivision' });
  }
});

// Update a subdivision by ID (admin-only)
router.put('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const { name, division } = req.body;
    const updatedSubdivision = await Subdivision.findByIdAndUpdate(
      req.params.id,
      { name, division },
      { new: true, runValidators: true }
    );
    if (!updatedSubdivision) {
      return res.status(404).json({ error: 'Subdivision not found' });
    }
    res.json(updatedSubdivision);
  } catch (error) {
    if (error.code === 11000) {
      return res.status(400).json({ error: 'Subdivision with this name already exists' });
    }
    res.status(500).json({ error: 'Failed to update subdivision' });
  }
});

// Delete a subdivision by ID (admin-only)
router.delete('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const subdivision = await Subdivision.findByIdAndDelete(req.params.id);
    if (!subdivision) {
      return res.status(404).json({ error: 'Subdivision not found' });
    }
    res.json({ message: 'Subdivision deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: 'Failed to delete subdivision' });
  }
});

export default router;