// routes/territories.js
import express from 'express';
import Territory from '../models/Territory.js';
import { authenticateToken, authorizeRole } from '../middleware/auth.js';

const router = express.Router();

// GET all territories
router.get('/', authenticateToken, async (req, res) => {
  try {
    const territories = await Territory.find({})
      .populate('division', 'name')
      .populate('subdivision', 'name');
    res.json(territories);
  } catch (error) {
    console.error('Error fetching territories:', error);
    res.status(500).json({ error: 'Failed to fetch territories' });
  }
});

// POST a new territory
router.post('/', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const { name, division, subdivision, geometry } = req.body;
    const newTerritory = new Territory({ name, division, subdivision, geometry });
    await newTerritory.save();
    res.status(201).json(newTerritory);
  } catch (error) {
    console.error('Error creating territory:', error);
    res.status(500).json({ error: 'Failed to create territory' });
  }
});

// PUT to update an existing territory
router.put('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const { name, division, subdivision, geometry } = req.body;
    const updatedTerritory = await Territory.findByIdAndUpdate(
      req.params.id,
      { name, division, subdivision, geometry },
      { new: true, runValidators: true }
    )
      .populate('division', 'name')
      .populate('subdivision', 'name');

    if (!updatedTerritory) {
      return res.status(404).json({ error: 'Territory not found' });
    }

    res.json(updatedTerritory);
  } catch (error) {
    console.error('Error updating territory:', error);
    res.status(500).json({ error: 'Failed to update territory' });
  }
});

// DELETE a territory
router.delete('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const territory = await Territory.findByIdAndDelete(req.params.id);

    if (!territory) {
      return res.status(404).json({ error: 'Territory not found' });
    }

    res.json({ message: 'Territory deleted successfully' });
  } catch (error) {
    console.error('Error deleting territory:', error);
    res.status(500).json({ error: 'Failed to delete territory' });
  }
});

export default router;