import express from 'express';
import fs from 'fs';
import multer from 'multer';
import { fileURLToPath } from 'url';
import path from 'path';
import PointType from '../models/PointType.js';
import { authenticateToken, authorizeRole } from '../middleware/auth.js';

const router = express.Router();

// Configure multer for file uploads
// Derive __dirname equivalent for ES modules
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadPath = path.join(__dirname, '../uploads/types');
    try {
      fs.mkdirSync(uploadPath, { recursive: true });
      cb(null, uploadPath);
    } catch (error) {
      console.error('Error creating upload directory:', error);
      cb(error, null);
    }
  },
  filename: (req, file, cb) => {
    if (!file.originalname) {
      return cb(new Error('File originalname is undefined'), null);
    }
    const uniqueSuffix = `${Date.now()}-${Math.round(Math.random() * 1e9)}`;
    cb(null, `icon-${uniqueSuffix}${path.extname(file.originalname)}`);
  }
});
const upload = multer({
  storage,
  fileFilter: (req, file, cb) => {
    if (!file.originalname) {
      return cb(new Error('File originalname is undefined'), false);
    }
    if (file.mimetype === 'image/svg+xml') {
      cb(null, true);
    } else {
      cb(new Error('Only SVG files are allowed'), false);
    }
  },
  limits: { fileSize: 5 * 1024 * 1024 } // 5MB limit
});

// Get all point types
router.get('/', authenticateToken, async (req, res) => {
  try {
    const types = await PointType.find({});
    res.json(types);
  } catch (error) {
    res.status(500).json({ error: 'Failed to fetch point types' });
  }
});

// Get single point type by ID
router.get('/:id', async (req, res) => {
  try {
    const type = await PointType.findById(req.params.id);
    if (!type) {
      return res.status(404).json({ error: 'Point type not found' });
    }
    res.json(type);
  } catch (error) {
    console.error('Error fetching point type:', error);
    res.status(500).json({ error: 'Failed to fetch point type' });
  }
});

// Create a point type (admin-only)
// Create a point type (admin-only)
router.post('/', authenticateToken, authorizeRole('Administrador'), upload.single('icon'), async (req, res) => {
  try {
    let { name } = req.body;
    let icon = null;

    if (req.file) {
      icon = `https://map-server.whirelab.com/uploads/types/${req.file.filename}`;
      console.log(`Uploaded file: ${req.file.filename}, URL: ${icon}`);
    }

    const newType = new PointType({
      name: name.trim(),
      icon
    });
    await newType.save();
    res.status(201).json(newType);
  } catch (error) {
    console.error('Create error:', error);
    if (req.file) {
      const filePath = path.join(__dirname, '../Uploads/types', req.file.filename);
      try {
        fs.unlinkSync(filePath);
      } catch (unlinkError) {
        console.error('Error deleting file:', unlinkError);
      }
    }
    if (error.code === 11000) {
      return res.status(400).json({ error: 'Point type with this name already exists' });
    }
    res.status(500).json({ error: 'Failed to create point type', details: error.message });
  }
});

// Update a point type by ID (admin-only)
router.put('/:id', authenticateToken, authorizeRole('Administrador'), upload.single('icon'), async (req, res) => {
  try {
    const { name } = req.body;
    const updatePayload = {};
    if (name) updatePayload.name = name.trim();

    if (req.file) {
       updatePayload.icon = `https://map-server.whirelab.com/uploads/types/${req.file.filename}`;
      //updatePayload.icon = `http://localhost:5000/uploads/types/${req.file.filename}`;
      console.log(`Uploaded file: ${req.file.filename}, URL: ${updatePayload.icon}`);
    }

    if (Object.keys(updatePayload).length === 0) {
      if (req.file) {
        const filePath = path.join(__dirname, '../Uploads/types', req.file.filename);
        try {
          fs.unlinkSync(filePath);
        } catch (unlinkError) {
          console.error('Error deleting file:', unlinkError);
        }
      }
      return res.status(400).json({ error: 'No valid fields provided for update' });
    }

    const updatedType = await PointType.findByIdAndUpdate(
      req.params.id,
      { $set: updatePayload },
      { new: true, runValidators: true }
    );
    if (!updatedType) {
      if (req.file) {
        const filePath = path.join(__dirname, '../Uploads/types', req.file.filename);
        try {
          fs.unlinkSync(filePath);
        } catch (unlinkError) {
          console.error('Error deleting file:', unlinkError);
        }
      }
      return res.status(404).json({ error: 'Point type not found' });
    }
    if (req.file && updatedType.icon && updatedType.icon !== updatePayload.icon) {
      const oldFilePath = path.join(__dirname, '../Uploads/types', path.basename(updatedType.icon));
      try {
        if (fs.existsSync(oldFilePath)) fs.unlinkSync(oldFilePath);
      } catch (unlinkError) {
        console.error('Error deleting old file:', unlinkError);
      }
    }
    res.json(updatedType);
  } catch (error) {
    console.error('Update error:', error);
    if (req.file) {
      const filePath = path.join(__dirname, '../Uploads/types', req.file.filename);
      try {
        fs.unlinkSync(filePath);
      } catch (unlinkError) {
        console.error('Error deleting file:', unlinkError);
      }
    }
    if (error.code === 11000) {
      return res.status(400).json({ error: 'Point type with this name already exists' });
    }
    if (error.name === 'ValidationError') {
      return res.status(400).json({
        error: 'Validation error',
        details: Object.values(error.errors).map(e => e.message),
      });
    }
    res.status(500).json({ error: 'Failed to update point type', details: error.message });
  }
});

// Delete a point type by ID (admin-only)
router.delete('/:id', authenticateToken, authorizeRole('Administrador'), async (req, res) => {
  try {
    const type = await PointType.findByIdAndDelete(req.params.id);
    if (!type) {
      return res.status(404).json({ error: 'Point type not found' });
    }
    if (type.icon) {
      const filePath = path.join(__dirname, '../uploads/types', path.basename(type.icon));
      if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    res.json({ message: 'Point type deleted successfully' });
  } catch (error) {
    console.error('Error deleting point type:', error);
    res.status(500).json({ error: 'Failed to delete point type' });
  }
});

export default router;