import express from 'express';
import mongoose from 'mongoose';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import dotenv from 'dotenv';
import pointsRouter from './routes/points.js';
import { authenticateToken } from './middleware/auth.js';
import { initializeFirebase } from './config/firebase.js';
import territoryRoutes from './routes/territories.js';
import typesRoutes from './routes/types.js';
import divisionsRoutes from './routes/divisions.js';
import subdivisionsRoutes from './routes/subdivisions.js';
dotenv.config();

const app = express();
const PORT = process.env.PORT || 5000;

// Security middleware
app.use(helmet());
app.use(cors({
  origin: process.env.NODE_ENV === 'production' 
    ? 'https://your-production-domain.com' // Replace with your domain
    : 'http://localhost:5173', // Correct port for your frontend
  credentials: true
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100 // limit each IP to 100 requests per windowMs
});
app.use(limiter);

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Initialize Firebase Admin
// This MUST happen before any Firebase service is used.
initializeFirebase();

// Now that Firebase is initialized, you can safely import the routes that depend on it.
import usersRouter from './routes/users.js';
import usersRouterSetup from './routes/users-setup.js';

// Connect to MongoDB
mongoose.connect(process.env.MONGODB_URI, {
  useNewUrlParser: true,
  useUnifiedTopology: true,
})
.then(() => console.log('Connected to MongoDB'))
.catch((error) => console.error('MongoDB connection error:', error));

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ 
    status: 'OK', 
    timestamp: new Date().toISOString(),
    database: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected'
  });
});

// Routes
app.use('/api/points', authenticateToken, pointsRouter);
app.use('/api/users', usersRouter); // Now it's safe to use the users router
app.use('/api/setup-admin', usersRouterSetup); // Main users route
app.use('/api/territories', territoryRoutes);
app.use('/api/types', typesRoutes);
app.use('/api/divisions', divisionsRoutes);
app.use('/api/subdivisions', subdivisionsRoutes);
// Error handling middleware
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
  console.log(`Environment: ${process.env.NODE_ENV || 'development'}`);
});

